const { app } = require('electron');
const { google } = require('googleapis');
const fs = require('fs');
const fsp = fs.promises;
const http = require('http');
const path = require('path');
const { URL } = require('url');
const crypto = require('crypto');

const accountStore = require('./youtubeAccountStore');
const logger = require('../logger');

const OAUTH_SERVER_HOST = '127.0.0.1';
const OAUTH_SERVER_PORT = 4587;
const REDIRECT_PATH = '/oauth2/callback';
const REDIRECT_URI = `http://${OAUTH_SERVER_HOST}:${OAUTH_SERVER_PORT}${REDIRECT_PATH}`;

const DEFAULT_SCOPES = Object.freeze([
  'https://www.googleapis.com/auth/youtube',
  'https://www.googleapis.com/auth/youtube.upload',
  'https://www.googleapis.com/auth/youtube.force-ssl'
]);

let cachedClientConfig = null;
let activeServer = null;
let activeServerPromise = null;
let activeServerOptions = null;
let activeAuthSession = null;

function resolveConfigPath() {
  const appPath = app.getAppPath();
  const resourcesPath = process.resourcesPath || appPath;
  const candidatePaths = [
    path.join(appPath, 'config', 'google-oauth.json'),
    path.join(resourcesPath, 'config', 'google-oauth.json')
  ];

  for (const candidate of candidatePaths) {
    if (fs.existsSync(candidate)) {
      return candidate;
    }
  }

  throw new Error('Google OAuth client configuration file not found (expected at config/google-oauth.json).');
}

async function loadClientConfig() {
  if (cachedClientConfig) {
    return cachedClientConfig;
  }

  const configPath = resolveConfigPath();
  const raw = await fsp.readFile(configPath, 'utf8');
  const parsed = JSON.parse(raw);
  const descriptor = parsed.installed || parsed.web || parsed;

  const clientId = descriptor.client_id;
  const clientSecret = descriptor.client_secret;

  if (!clientId || !clientSecret) {
    throw new Error('Google OAuth client configuration is missing client_id or client_secret.');
  }

  cachedClientConfig = {
    clientId,
    clientSecret
  };
  return cachedClientConfig;
}

async function createBaseOAuthClient() {
  const { clientId, clientSecret } = await loadClientConfig();
  return new google.auth.OAuth2(clientId, clientSecret, REDIRECT_URI);
}

function toBase64Url(buffer) {
  return buffer.toString('base64')
    .replace(/\+/gu, '-')
    .replace(/\//gu, '_')
    .replace(/=+$/u, '');
}

function generateCodeVerifier() {
  // Generates a verifier between 64-96 characters composed of URL-safe characters.
  const entropy = crypto.randomBytes(64);
  const verifier = toBase64Url(entropy);
  return verifier.length >= 43 ? verifier.slice(0, 96) : `${verifier}${toBase64Url(crypto.randomBytes(32))}`.slice(0, 96);
}

function generateCodeChallenge(verifier) {
  return toBase64Url(crypto.createHash('sha256').update(verifier).digest());
}

function generateStateToken() {
  return toBase64Url(crypto.randomBytes(32));
}

function createTransientAuthSession() {
  if (activeAuthSession) {
    throw new Error('An OAuth authentication session is already active.');
  }
  const codeVerifier = generateCodeVerifier();
  const session = {
    id: typeof crypto.randomUUID === 'function' ? crypto.randomUUID() : toBase64Url(crypto.randomBytes(16)),
    codeVerifier,
    codeChallenge: generateCodeChallenge(codeVerifier),
    state: generateStateToken(),
    createdAt: Date.now()
  };
  activeAuthSession = session;
  return session;
}

function getActiveAuthSession(sessionId) {
  if (!activeAuthSession || (sessionId && activeAuthSession.id !== sessionId)) {
    return null;
  }
  return activeAuthSession;
}

function clearActiveAuthSession(sessionId) {
  if (activeAuthSession && (!sessionId || activeAuthSession.id === sessionId)) {
    activeAuthSession = null;
  }
}

function normalizeScopes(scopes) {
  const requested = Array.isArray(scopes) ? scopes.filter(Boolean) : [];
  const merged = new Set([...DEFAULT_SCOPES, ...requested]);
  return Array.from(merged);
}

function attachTokenListener(oauthClient, accountId) {
  if (!accountId) {
    return;
  }
  oauthClient.on('tokens', (tokens) => {
    if (!tokens) {
      return;
    }
    accountStore.mergeAccountTokens(accountId, tokens).catch((error) => {
      logger.warn('Failed to persist refreshed YouTube tokens:', error && error.message ? error.message : error);
    });
  });
}

async function createOAuthClient(accountId) {
  const oauthClient = await createBaseOAuthClient();
  const storedTokens = await accountStore.getAccountTokens(accountId);
  if (!storedTokens || (!storedTokens.access_token && !storedTokens.refresh_token)) {
    throw new Error('YouTube account is not connected.');
  }
  oauthClient.setCredentials(storedTokens);
  const resolvedAccountId = accountId || (await accountStore.getActiveAccountId());
  attachTokenListener(oauthClient, resolvedAccountId);
  return { oauthClient, accountId: resolvedAccountId };
}

async function ensureFreshCredentials(oauthClient, accountId) {
  try {
    const accessTokenResponse = await oauthClient.getAccessToken();
    if (!accessTokenResponse || !accessTokenResponse.token) {
      throw new Error('Failed to obtain access token from refresh token.');
    }
    await accountStore.mergeAccountTokens(accountId, oauthClient.credentials || {});
    return oauthClient.credentials;
  } catch (error) {
    if (error && error.code === 401) {
      throw new Error('Stored credentials are no longer valid. Please re-authenticate.');
    }
    throw error;
  }
}

async function fetchPrimaryChannelProfile(oauthClient) {
  try {
    const youtube = google.youtube({
      version: 'v3',
      auth: oauthClient
    });
    const response = await youtube.channels.list({
      mine: true,
      part: ['id', 'snippet']
    });
    const items = response && response.data && Array.isArray(response.data.items)
      ? response.data.items
      : [];
    if (items.length === 0) {
      return null;
    }
    const channel = items[0];
    const snippet = channel && channel.snippet ? channel.snippet : {};
    const thumbnails = snippet.thumbnails || {};
    const thumbnailUrl = thumbnails.high?.url || thumbnails.medium?.url || thumbnails.default?.url || null;
    return {
      channelId: channel.id || null,
      channelTitle: snippet.title || null,
      channelCustomUrl: snippet.customUrl || null,
      channelThumbnailUrl: thumbnailUrl
    };
  } catch (error) {
      logger.warn('Failed to fetch YouTube channel profile during authentication:', error && error.message ? error.message : error);
    return null;
  }
}

async function registerAuthenticatedAccount(oauthClient, scopes) {
  const normalizedScopes = normalizeScopes(scopes);
  const profile = await fetchPrimaryChannelProfile(oauthClient);
  const tokens = oauthClient.credentials || {};
  const now = new Date().toISOString();

  let accountId = profile && profile.channelId ? profile.channelId : null;
  if (!accountId) {
    // Fallback to hashing refresh token or timestamp to guarantee uniqueness.
    const refreshToken = tokens.refresh_token || '';
    const hash = refreshToken
      ? Buffer.from(refreshToken).toString('base64').replace(/\+/g, '-').replace(/\//g, '_').replace(/=+$/u, '').slice(0, 16)
      : `acct-${Date.now()}`;
    accountId = hash;
  }

  const accountPayload = {
    id: accountId,
    channelId: profile ? profile.channelId : null,
    channelTitle: profile ? profile.channelTitle : null,
    channelCustomUrl: profile ? profile.channelCustomUrl : null,
    channelThumbnailUrl: profile ? profile.channelThumbnailUrl : null,
    tokens,
    scopes: normalizedScopes,
    lastUsedAt: now
  };

  const savedAccount = await accountStore.upsertAccount(accountPayload);
  await accountStore.setActiveAccount(savedAccount.id);
  await accountStore.markAccountUsed(savedAccount.id);
  return savedAccount;
}

function startLocalCallbackServer(options = {}) {
  if (activeServerPromise) {
    return activeServerPromise;
  }

  const { expectedState = null } = options;
  activeServerOptions = { expectedState };

  activeServerPromise = new Promise((resolve, reject) => {
    const server = http.createServer((req, res) => {
      try {
        const requestUrl = new URL(req.url, `http://${req.headers.host}`);
        if (requestUrl.pathname !== REDIRECT_PATH) {
          res.writeHead(404, { 'Content-Type': 'text/plain' });
          res.end('Not Found');
          return;
        }

        const errorParam = requestUrl.searchParams.get('error');
        const codeParam = requestUrl.searchParams.get('code');
        const stateParam = requestUrl.searchParams.get('state');

        if (errorParam) {
          res.writeHead(200, { 'Content-Type': 'text/html; charset=utf-8' });
          res.end('<html><body><h2>YouTube authentication failed.</h2><p>You can close this window.</p></body></html>');
          resolve({ error: new Error(`Authorization failed: ${errorParam}`) });
          shutdownLocalCallbackServer().catch(() => {});
          return;
        }

        if (!codeParam) {
          res.writeHead(400, { 'Content-Type': 'text/plain' });
          res.end('Missing authorization code.');
          return;
        }

        if (!stateParam || !expectedState || stateParam !== expectedState) {
          res.writeHead(400, { 'Content-Type': 'text/plain' });
          res.end('Invalid OAuth state parameter.');
          resolve({ error: new Error('Invalid OAuth state parameter.') });
          shutdownLocalCallbackServer().catch(() => {});
          return;
        }

        res.writeHead(200, { 'Content-Type': 'text/html; charset=utf-8' });
        res.end('<html><body><h2>YouTube account connected.</h2><p>You can close this window and return to the app.</p></body></html>');
        resolve({ code: codeParam, state: stateParam });
        shutdownLocalCallbackServer().catch(() => {});
      } catch (error) {
        res.writeHead(500, { 'Content-Type': 'text/plain' });
        res.end('Internal error handling OAuth callback.');
        reject(error);
        shutdownLocalCallbackServer().catch(() => {});
      }
    });

    server.on('error', (error) => {
      reject(error);
      shutdownLocalCallbackServer().catch(() => {});
    });

    server.listen(OAUTH_SERVER_PORT, OAUTH_SERVER_HOST, () => {
      activeServer = server;
    });
  });

  return activeServerPromise;
}

async function shutdownLocalCallbackServer() {
  if (activeServer) {
    await new Promise((resolve) => {
      activeServer.close(() => resolve());
    });
  }
  activeServer = null;
  activeServerPromise = null;
  activeServerOptions = null;
}

async function beginAuthentication(scopes = DEFAULT_SCOPES) {
  const normalizedScopes = normalizeScopes(scopes);
  const oauthClient = await createBaseOAuthClient();
  const authSession = createTransientAuthSession();
  const authUrl = oauthClient.generateAuthUrl({
    access_type: 'offline',
    scope: normalizedScopes,
    prompt: 'consent',
    redirect_uri: REDIRECT_URI,
    code_challenge: authSession.codeChallenge,
    code_challenge_method: 'S256',
    state: authSession.state
  });

  const serverResultPromise = startLocalCallbackServer({
    expectedState: authSession.state
  });

  return {
    authUrl,
    async waitForTokens(timeoutMs = 3 * 60 * 1000) {
      let timer = null;
      const timeoutPromise = new Promise((_, reject) => {
        timer = setTimeout(() => {
          timer = null;
          reject(new Error('Timed out waiting for Google authentication.'));
          shutdownLocalCallbackServer().catch(() => {});
        }, timeoutMs);
      });

      let raceResult;
      try {
        raceResult = await Promise.race([serverResultPromise, timeoutPromise]);
      } catch (error) {
        clearActiveAuthSession(authSession.id);
        throw error;
      }
      if (timer) {
        clearTimeout(timer);
        timer = null;
      }
      if (!raceResult) {
        throw new Error('Unexpected empty OAuth result.');
      }
      if (raceResult.error) {
        clearActiveAuthSession(authSession.id);
        throw raceResult.error;
      }

      try {
        const tokenResponse = await oauthClient.getToken({
          code: raceResult.code,
          codeVerifier: authSession.codeVerifier
        });
        oauthClient.setCredentials(tokenResponse.tokens);
        const account = await registerAuthenticatedAccount(oauthClient, normalizedScopes);
        attachTokenListener(oauthClient, account.id);
        return {
          account,
          tokens: oauthClient.credentials
        };
      } finally {
        clearActiveAuthSession(authSession.id);
      }
    },
    async cancel() {
      clearActiveAuthSession(authSession.id);
      await shutdownLocalCallbackServer();
    }
  };
}

async function getAuthenticatedClient(scopes = DEFAULT_SCOPES, options = {}) {
  const normalizedScopes = normalizeScopes(scopes);
  const requestedAccountId = options && typeof options === 'object' ? options.accountId : null;
  const { oauthClient, accountId } = await createOAuthClient(requestedAccountId);

  try {
    const existingAccount = await accountStore.getAccount(accountId);
    if (existingAccount) {
      const existingScopes = new Set(Array.isArray(existingAccount.scopes) ? existingAccount.scopes : []);
      let scopesChanged = false;
      for (const scope of normalizedScopes) {
        if (!existingScopes.has(scope)) {
          existingScopes.add(scope);
          scopesChanged = true;
        }
      }
      if (scopesChanged) {
        await accountStore.upsertAccount({
          ...existingAccount,
          id: existingAccount.id,
          tokens: existingAccount.tokens,
          scopes: Array.from(existingScopes),
          lastUsedAt: existingAccount.lastUsedAt
        });
      }
    }
  } catch (error) {
    logger.warn('Failed to persist YouTube account scopes:', error && error.message ? error.message : error);
  }

  await ensureFreshCredentials(oauthClient, accountId);
  await accountStore.markAccountUsed(accountId);
  oauthClient.youtubeAccountId = accountId;
  return oauthClient;
}

async function getAuthStatus() {
  const snapshot = await accountStore.getStoreSnapshot();
  const accounts = Object.values(snapshot.accounts || {}).map((account) => {
    const tokens = account.tokens || {};
    const expiryDate = tokens.expiry_date ? new Date(tokens.expiry_date).toISOString() : null;
    return {
      id: account.id,
      channelId: account.channelId,
      channelTitle: account.channelTitle,
      channelCustomUrl: account.channelCustomUrl,
      channelThumbnailUrl: account.channelThumbnailUrl,
      expiresAt: expiryDate,
      hasRefreshToken: Boolean(tokens.refresh_token),
      scopes: Array.isArray(account.scopes) ? [...account.scopes] : [],
      lastUsedAt: account.lastUsedAt || null
    };
  });

  const activeAccount = snapshot.activeAccountId ? snapshot.accounts[snapshot.activeAccountId] : null;
  const activeTokens = activeAccount && activeAccount.tokens ? activeAccount.tokens : null;
  const expiresAt = activeTokens && activeTokens.expiry_date
    ? new Date(activeTokens.expiry_date).toISOString()
    : null;

  return {
    authenticated: Boolean(activeAccount),
    activeAccountId: snapshot.activeAccountId || null,
    expiresAt,
    hasRefreshToken: Boolean(activeTokens && activeTokens.refresh_token),
    accounts
  };
}

async function logout(accountId = null) {
  const targetAccountId = accountId || await accountStore.getActiveAccountId();
  if (!targetAccountId) {
    await accountStore.clearStore();
    return;
  }

  const account = await accountStore.getAccount(targetAccountId);
  if (!account || !account.tokens) {
    await accountStore.removeAccount(targetAccountId);
    return;
  }

  const oauthClient = await createBaseOAuthClient();
  oauthClient.setCredentials({ ...account.tokens });

  const accessToken = account.tokens.access_token;
  const refreshToken = account.tokens.refresh_token;

  if (accessToken) {
    try {
      await oauthClient.revokeToken(accessToken);
    } catch (error) {
      // Ignore failures while revoking; still clear local state.
    }
  }
  if (refreshToken) {
    try {
      await oauthClient.revokeToken(refreshToken);
    } catch (error) {
      // Ignore failures.
    }
  }

  await accountStore.removeAccount(targetAccountId);
}

async function listAccounts() {
  return accountStore.listAccounts();
}

async function setActiveAccount(accountId) {
  return accountStore.setActiveAccount(accountId);
}

async function removeAccount(accountId) {
  return logout(accountId);
}

async function updateAccountProfile(accountId, profile = {}) {
  if (!accountId) {
    return null;
  }
  const existingAccount = await accountStore.getAccount(accountId);
  if (!existingAccount) {
    return null;
  }
  const payload = {
    ...existingAccount,
    id: existingAccount.id,
    tokens: existingAccount.tokens,
    channelId: profile.channelId != null ? profile.channelId : existingAccount.channelId,
    channelTitle: profile.channelTitle != null ? profile.channelTitle : existingAccount.channelTitle,
    channelCustomUrl: profile.channelCustomUrl != null ? profile.channelCustomUrl : existingAccount.channelCustomUrl,
    channelThumbnailUrl: profile.channelThumbnailUrl != null ? profile.channelThumbnailUrl : existingAccount.channelThumbnailUrl,
    lastUsedAt: existingAccount.lastUsedAt
  };
  return accountStore.upsertAccount(payload);
}

async function readStoredTokens(accountId = null) {
  return accountStore.getAccountTokens(accountId);
}

async function writeStoredTokens(tokens, accountId = null) {
  const targetId = accountId || await accountStore.getActiveAccountId();
  if (!targetId) {
    throw new Error('Unable to persist tokens because no active YouTube account is selected.');
  }
  return accountStore.mergeAccountTokens(targetId, tokens);
}

async function deleteStoredTokens(accountId = null) {
  const targetId = accountId || await accountStore.getActiveAccountId();
  if (!targetId) {
    await accountStore.clearStore();
    return;
  }
  await accountStore.removeAccount(targetId);
}

module.exports = {
  DEFAULT_SCOPES,
  REDIRECT_URI,
  OAUTH_SERVER_PORT,
  beginAuthentication,
  getAuthenticatedClient,
  getAuthStatus,
  logout,
  listAccounts,
  setActiveAccount,
  removeAccount,
  updateAccountProfile,
  readStoredTokens,
  writeStoredTokens,
  deleteStoredTokens
};

